# coding: utf-8
# NAME: Color Deconvolution conversion
# FILE: ColorDec_Rev2
# REVISION : 1.1.0 - 01/07/2021
# AUTHOR : Maurizio Abbate
# Copyright(c) 2021 arivis AG, Germany. All Rights Reserved.
#
# Permission is granted to use, modify and distribute this code,
# as long as this copyright notice remains part of the code.
#
# PURPOSE : Convert a RGB image in  Stain colors
#
# Tested for V4d Release : 3.4 
#
# NOTE: 
#   This script implements the method described in the following paper:
#   Ruifrok, A.C. & Johnston, D.A. (2001), 
#   "Quantification of histochemical staining by color deconvolution", 
#   Anal. Quant. Cytol. Histol. 23: 291-299, PMID 11531144
#
#   Landini G, Martinelli G, Piccinini F. Colour Deconvolution
#    – stain unmixing in histological imaging. Bioinformatics 2020,
#    https://doi.org/10.1093/bioinformatics/btaa847
# ------------------------------ External Package Import ----------------------
import math as Math
import time
import arivis
import arivis_objects as objects
import arivis_core as core
import numpy as np
# ----------------------  End of external Package Import ---------------------
#
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ USER SETTINGS @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# -----------------------------------------------------------------------------
# Skeleton section - Script
# INPUT_CHANNEL channel with the binary image
#      is only used to read/write the voxels 
# ---> imageset.get_channeldata / imageset.set_channeldata
# -----------------------------------------------------------------------------
INPUT_CHANNEL_R = 0
INPUT_CHANNEL_G = 1
INPUT_CHANNEL_B = 2
# -------------------------------------------------------------------------
#   0 == HEMA_DAB 
#   1 == HEMA_EOSIN 
#   2 == HEMA_EOSIN2  
#   3 == FEULGEN_LGRENN
#   4 == HEMA_EOSIN_DAB
#   5 == GIEMSA  
#   6 == Fast Red, Fast Blue and DAB
#   7 == Methyl green and DAB
#   8 == Haematoxylin, Eosin and DAB (H&E DAB)
#   9 == Haematoxylin and AEC (H AEC) 
#  10 == Azan-Mallory
#  11 == Alcian blue & Haematoxylin
#  12 == MASSON THRICROME
#  13 == Haematoxylin and Periodic Acid of Schiff (PAS)
# -------------------------------------------------------------------------
STAINING = 0
# -----------------------------------------------------------------------------
INVERT_IMAGE = True
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ END USER SETTINGS @@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# ------------------------------ Global variables  ---------------------------- 
#   WARNING .......  DON'T MODIFY the following variables
# -----------------------------------------------------------------------------
#
TILESIZE_X = 2048
TILESIZE_Y = 2048
TILE_SIZE = core.Size2D(TILESIZE_X,TILESIZE_Y)
#
# ----------------------  End of the Global variables  ------------------------
#
# @@@@@@@@@@@@@@@@@@@@@@@@@@ V4D Utility Functions @@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# ------------------------------ Script body ---------------------------------- 
# Function : GetEnviroment
# -----------------------------------------------------------------------------
def GetEnviroment():
  # ---------------------------------------------------------------------------
  # return the viewer and the imageset objects
  # ---------------------------------------------------------------------------
  viewer = arivis.App.get_active_viewer()
  imageset = viewer.get_imageset()
  if imageset is None :             #  if None == imageset :
    print("No Image Set open") 
    return viewer,None  
  # ---------------------------------------------------------------------------
  return viewer,imageset
# ------------------------------ Script body ----------------------------------
# End Function : GetEnviroment
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : GetStorage
# -----------------------------------------------------------------------------
def GetStorage(imageset,bObject = False):
  # ---------------------------------------------------------------------------
  # return the storage object
  # ---------------------------------------------------------------------------
  if imageset is None :
    print("No Image Set open") 
    return None  
  # ---------------------------------------------------------------------------
  document = imageset.get_document()
  if document is None:  # None == document :
    print( "No Document open" )
    return None
  # ---------------------------------------------------------------------------
  store = document.get_store(imageset, objects.Store.DOCUMENT_STORE)
  if store is None:  #None == store :
    print ("No Measurements availble" )
    return None
  # ---------------------------------------------------------------------------
  # Get the full object's ID list - if the list's lenght is == 0 no storage 
  # ---------------------------------------------------------------------------
  IdList = store.get_object_ids("")
  #print "lenght " + str(len(IdList))   
  #----------------------------------------------------------------------------  
  if 0 == len(IdList) and bObject == True:
    print ("No Objects" )
    return None
  # ---------------------------------------------------------------------------
  print ("Get_Storage - OK " )
  return store
# ------------------------------ Script body ----------------------------------
# End Function : GetStorage
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : GetDocument
# -----------------------------------------------------------------------------
def GetDocument(imageset):
  # ---------------------------------------------------------------------------
  # return the document object
  # ---------------------------------------------------------------------------
  if imageset is None:
    print ("No Image Set open" )
    return None  
  # ---------------------------------------------------------------------------
  document = imageset.get_document()
  # ---------------------------------------------------------------------------
  if document is None:
    print ("No Document open" )
    return None
  # ---------------------------------------------------------------------------
  return document
# ------------------------------ Script body ----------------------------------
# End Function : GetDocument
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : GetObject
# -----------------------------------------------------------------------------
def GetObject(store,IdOBJ):
  # ---------------------------------------------------------------------------
  # return the object centroid (3D_Bounding Box - pixels)
  # NO_Error , Point3D(X ,Y, Z)
  # strTAG
  # ---------------------------------------------------------------------------
  if store is None :
   print ("No Measurements available" )
   return None 
  # ---------------------------------------------------------------------------
  object1 = store.get_object(IdOBJ,True) 
  if object1 is None:  
    # -------------------------------------------------------------------------
    print ("No Segment available" )
    return None
  else:
    return object1
# ------------------------------ Script body ---------------------------------- 
# End Function : GetObject
# -----------------------------------------------------------------------------
#   
# ----------------------------------------------------------------------------- 
# Function : PrepareChannel
# new channel - return channel index
# -----------------------------------------------------------------------------  
def PrepareChannel(imageset,chname):
  # --------------------------------------------------------------------------- 
  # Channel(s) is added at the end
  # --------------------------------------------------------------------------- 
  channels = imageset.get_channel_count()
  print( "channels " + str(channels))
  imageset.insert_channels(channels,1)
  if chname!="":
      lastCh = imageset.get_channel_count()
      imageset.set_channel_name(lastCh - 1,chname + str(lastCh))
  return channels 
# ----------------------------------------------------------------------------- 
# End Function : PrepareChannel
# -----------------------------------------------------------------------------
#
# ----------------------------------------------------------------------------- 
# Function : GetPixelTypes()
# -----------------------------------------------------------------------------
def GetPixelTypes(imageset):
  # -------------------------------------------------------------------------
  # ritorna coefficente, dt, Type
  # -------------------------------------------------------------------------    
  if imageset is None :
    print( "No Image Set open" )
    return None  
  # -------------------------------------------------------------------------   
  pixelType = imageset.get_pixeltype()
  # -------------------------------------------------------------------------  
  if pixelType == core.ImageSet.PIXELTYPE_USHORT:
    typeP = 'H'    
    #dt = np.ushort
    dt = np.dtype(np.ushort) 
    Coefficent = int(65535)   
  elif pixelType == core.ImageSet.PIXELTYPE_ULONG:
    typeP = 'L' 
    #dt = np.uint
    dt = np.dtype(np.uint)
    Coefficent = int(65535)
  elif pixelType == core.ImageSet.PIXELTYPE_FLOAT:
    typeP = 'f' 
    #dt = np.float32
    dt = np.dtype(np.float32)
    Coefficent = int(65535)
  else:     #core.ImageSet.PIXELTYPE_UCHAR:
    Coefficent = int(255)
    typeP = 'B'
    #dt = np.uint8
    dt = np.dtype(np.uint8)
  # -------------------------------------------------------------------------
  return dt,typeP,Coefficent 
# ----------------------------------------------------------------------------- 
# End Function : GetPixelTypes()
# -----------------------------------------------------------------------------
#
# @@@@@@@@@@@@@@@@@@@@@@@@@@ Color Dec Functions @@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# ----------------------------------------------------------------------------- 
# Function : StainMatrix()
# Private dModX(3) As Double
# Private dModY(3) As Double
# Private dModZ(3) As Double
# -----------------------------------------------------------------------------
def StainMatrix(stain):
    # -------------------------------------------------------------------------
    #   HEMA_DAB = 0  
    #   HEMA_EOSIN = 1  
    #   HEMA_EOSIN2 = 2  
    #   FEULGEN_LGRENN = 3  
    #   HEMA_EOSIN_DAB = 4
    #   GIEMSA = 5  
    #   Fast Red, Fast Blue and DAB = 6
    #   Methyl green and DAB = 7  
    #   Haematoxylin, Eosin and DAB (H&E DAB) = 8
    #   Haematoxylin and AEC (H AEC) = 9  
    #   Azan-Mallory = 10
    #   Alcian blue & Haematoxylin = 11 
    #   MASSON THRICROME  = 12
    #   Haematoxylin and Periodic Acid of Schiff (PAS) = 13
    # -------------------------------------------------------------------------
    Stain1=[]
    Stain2=[]
    Stain3=[]    
    # -------------------------------------------------------------------------
    if stain < 0:
        print("Stain not defined")
        return Stain1,Stain2,Stain3
    # -------------------------------------------------------------------------    
    # 3,3-diamino-benzidine tetrahydrochloride
    # ------------------------------------------------------------------------- 
    if stain == 0 :      #HEMA_DAB
        #Haem matrix
        Stain1.append(0.65)
        Stain2.append(0.704)
        Stain3.append(0.286)    
        #DAB matrix   
        Stain1.append(0.268)
        Stain2.append(0.570)
        Stain3.append(0.776)  
        #Zero matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0) 
    elif stain == 1 :    #HEMA_EOSIN        
        #Haem matrix
        Stain1.append(0.644211000)
        Stain2.append(0.716556000)
        Stain3.append(0.26684400)    
        #EOS matrix   
        Stain1.append(0.09278900)
        Stain2.append(0.95411100)
        Stain3.append(0.28311100)  
        #Zero matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0) 
    elif stain == 2 :    #HEMA_EOSIN2       
        #Haem matrix
        Stain1.append(0.490157340)
        Stain2.append(0.768970850)
        Stain3.append(0.410401730)    
        #EOS matrix   
        Stain1.append(0.04615336)
        Stain2.append(0.84206840)
        Stain3.append(0.53739250)  
        #Zero matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0) 
    elif stain == 3 :    #FEULGEN_LGRENN
        #Feulgen matrix
        Stain1.append(0.464209200)
        Stain2.append(0.830083350)
        Stain3.append(0.830083350)    
        #Light Green matrix  
        Stain1.append(0.94705542)
        Stain2.append(0.25373821)
        Stain3.append(0.19650764)  
        #Zero matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)         
    elif stain == 4 :    #HEMA_EOSIN_DAB  
        #Haem matrix
        Stain1.append(0.650000000)
        Stain2.append(0.704000000)
        Stain3.append(0.286000000)    
        #EOS matrix  
        Stain1.append(0.07200000)
        Stain2.append(0.99000000)
        Stain3.append(0.10500000)  
        #DAB matrix 
        Stain1.append(0.26800000)
        Stain2.append(0.57000000)
        Stain3.append(0.7760000)      
    elif stain == 5 :    # GIEMSA    
        Stain1.append(0.83475024)
        Stain2.append(0.5135563)
        Stain3.append(0.1963304)    
        #EOS matrix  
        Stain1.append(0.092789)
        Stain2.append(0.954111)
        Stain3.append(0.283111)  
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)           
    elif stain == 6 :    # Fast Red, Fast Blue and DAB    
        Stain1.append(0.2139392)
        Stain2.append(0.8511267)
        Stain3.append(0.47794023)    
        #EOS matrix  
        Stain1.append(0.7489029)
        Stain2.append(0.6062416)
        Stain3.append(0.26731083)  
        #DAB matrix 
        Stain1.append(0.268)
        Stain2.append(0.57)
        Stain3.append(0.776)        
    elif stain == 7 :    # Methyl green and DAB   
        Stain1.append(0.98003)
        Stain2.append(0.144316)
        Stain3.append(0.133146)    
        #EOS matrix  
        Stain1.append(0.268)
        Stain2.append(0.57)
        Stain3.append(0.776)  
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)                        
    elif stain == 8 :    # Haematoxylin, Eosin and DAB (H&E DAB)  
        Stain1.append(0.65)
        Stain2.append(0.704)
        Stain3.append(0.286)   
        #EOS matrix          
        Stain1.append(0.072)
        Stain2.append(0.99)
        Stain3.append(0.105)   
        #DAB matrix 
        Stain1.append(0.268)
        Stain2.append(0.57)
        Stain3.append(0.776)             
    elif stain == 9:     # Haematoxylin and AEC (H AEC) 
        Stain1.append(0.65)
        Stain2.append(0.704)
        Stain3.append(0.286)   
        #EOS matrix          
        Stain1.append(0.2743)
        Stain2.append(0.6796)
        Stain3.append(0.6803)   
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)           
    elif stain == 10 :    # Azan-Mallory
        Stain1.append(0.853033)
        Stain2.append(0.508733)
        Stain3.append(0.112656)   
        #EOS matrix          
        Stain1.append(0.09289875)
        Stain2.append(0.8662008)
        Stain3.append(0.49098468)   
        #DAB matrix 
        Stain1.append(0.10732849)
        Stain2.append(0.36765403)
        Stain3.append(0.9237484)          
    elif stain == 11 :    # Alcian blue & Haematoxylin
        Stain1.append(0.874622)
        Stain2.append(0.457711)
        Stain3.append(0.158256)   
        #EOS matrix          
        Stain1.append(0.552556)
        Stain2.append(0.7544)
        Stain3.append(0.353744)   
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)         
    elif stain == 12 :   # MASSON THRICROME
        Stain1.append(0.7995107)
        Stain2.append(0.5913521)
        Stain3.append(0.10528667)   
        #EOS matrix          
        Stain1.append(0.09997159)
        Stain2.append(0.73738605)
        Stain3.append(0.6680326)   
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0) 
    elif stain == 13 :   # Haematoxylin and Periodic Acid of Schiff (PAS)
        Stain1.append(0.644211)
        Stain2.append(0.716556)
        Stain3.append(0.266844)   
        #EOS matrix          
        Stain1.append(0.175411)
        Stain2.append(0.972178)
        Stain3.append(0.154589)   
        #DAB matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)        
    else:
        print("Stain not supported - return the HEMA_DAB (0) instead : " + str(stain))
        #Haem matrix
        Stain1.append(0.65)
        Stain2.append(0.704)
        Stain3.append(0.286)    
        #DAB matrix   
        Stain1.append(0.268)
        Stain2.append(0.570)
        Stain3.append(0.776)  
        #Zero matrix 
        Stain1.append(0.0)
        Stain2.append(0.0)
        Stain3.append(0.0)                                                                                                           
    # ------------------------------------------------------------------------- 
    return Stain1,Stain2,Stain3   
# ----------------------------------------------------------------------------- 
# End Function : StainMatrix()
# -----------------------------------------------------------------------------
#    
# ----------------------------------------------------------------------------- 
# Function : NormalizeVectorLength (NormalizzaLunghezzaVettori())
# -----------------------------------------------------------------------------
def NormalizeVectorLength(Stain1,Stain2,Stain3):
    # -------------------------------------------------------------------------
    dCosX = []
    dCosY = []
    dCosZ = []   
    dLen = []
    # -------------------------------------------------------------------------
    for i in range(0,3):
        # ---------------------------------------------------------------------
        dCosX.append(0.0)
        dCosY.append(0.0)
        dCosZ.append(0.0)
        dPowX = Math.pow(Stain1[i],2.0)
        dPowY = Math.pow(Stain2[i],2.0)
        dPowZ = Math.pow(Stain3[i],2.0)  
        dLen.append(dPowX+dPowY+dPowZ)
        # ---------------------------------------------------------------------
        if dPowX+dPowY+dPowZ !=0.0:
            dCosX[i]=Stain1[i]/dLen[i]
            dCosY[i]=Stain2[i]/dLen[i]
            dCosZ[i]=Stain3[i]/dLen[i]
    # ---------------------------------------------------------------------
    return dCosX,dCosY,dCosZ,dLen  
# ----------------------------------------------------------------------------- 
# End Function : NormalizeVectorLength()
# -----------------------------------------------------------------------------
# 
# ----------------------------------------------------------------------------- 
# Function : TranslationMatrix 
# -----------------------------------------------------------------------------
def TranslationMatrix(dCosX,dCosY,dCosZ):
    # ---------------------------------------------------------------------
    if dCosX[1] == 0.0:
        if dCosY[1] == 0.0:  
            if dCosZ[1] == 0.0: 
                dCosX[1]=dCosZ[0]
                dCosY[1]=dCosX[0]
                dCosZ[1]=dCosY[0]   
    # ---------------------------------------------------------------------
    dPow = 0.0
    dPow1 = 0.0
    if dCosX[2] == 0.0 :       
        if dCosY[2] == 0.0:  
            if dCosZ[2] == 0.0:  
                dPow = Math.pow(dCosX[0],2.0 ) 
                dPow1 = Math.pow(dCosX[1],2.0 )   
                #-------------------------------------------------------------
                if dPow+dPow1 > 1.0 :
                    #-------------------------------------------------------------
                    dCosX[2] = 0.0
                    #-------------------------------------------------------------
                else:
                    #-------------------------------------------------------------
                    dCosX[2] = Math.pow(1.0-dPow-dPow1,0.5)
                     #-------------------------------------------------------------
                dPow = Math.pow(dCosZ[0],2.0)
                dPow1 = Math.pow(dCosZ[1],2.0)
                #-------------------------------------------------------------
                if dPow+dPow1 > 1.0:
                    #-------------------------------------------------------------
                    dCosZ[2]=0.0
                    #-------------------------------------------------------------
                else:
                    #-------------------------------------------------------------
                    dCosZ[2] = Math.pow(1.0-dPow-dPow1,0.5)
    #-------------------------------------------------------------
    dPowX = 0.0
    dPowY = 0.0
    dPowZ = 0.0  
    #-------------------------------------------------------------
    dPowX = Math.pow(dCosX[2],2.0)
    dPowY = Math.pow(dCosY[2],2.0)
    dPowZ = Math.pow(dCosY[2],2.0)
    dLeng = Math.pow(dPowX+dPowY+dPowZ,0.5)
    if dLeng > 0.0 :
    #-------------------------------------------------------------
        dCosX[2] = dCosX[2]/dLeng
        dCosY[2] = dCosY[2]/dLeng
        dCosZ[2] = dCosZ[2]/dLeng
    #-------------------------------------------------------------
    for i in range(0,3):
        #-------------------------------------------------------------
        if dCosX[i] == 0.0 :
            dCosX[i] = 0.001
        if dCosY[i] == 0.0 :
            dCosY[i] = 0.001
        if dCosZ[i] == 0.0:
            dCosZ[i] = 0.001
        #-------------------------------------------------------------
    return dCosX,dCosY,dCosZ                      
# ----------------------------------------------------------------------------- 
# End Function : TranslationMatrix
# -----------------------------------------------------------------------------
# 
# ----------------------------------------------------------------------------- 
# Function : ConvertPixel 
# return  bRed , bGreen , bBlu pixel
# dQ(9)
# -----------------------------------------------------------------------------
def ConvertPixel(bRed , bGreen , bBlu, dQ):
    #------------------------------------------------------------- 
    bRed1 = 0
    bGreen1 = 0
    bBlu1 = 0
    #------------------------------------------------------------- 
    if len(dQ) != 9:
        return bRed1 , bGreen1 , bBlu1  
    #-------------------------------------------------------------
    dLog255 = Math.log(255.0) 
    dRlog = -((255.0*Math.log((bRed+1)/255.0))/dLog255)
    dGlog = -((255.0*Math.log((bGreen+1)/255.0))/dLog255)
    dBlog = -((255.0*Math.log((bBlu+1)/255.0))/dLog255)    
    #-------------------------------------------------------------
    for iIndice in range (0,3):
        # rescale To match original paper values
        dRscaled = dRlog * dQ[iIndice*3]
        dGscaled = dGlog * dQ[(iIndice*3)+1]
        dBscaled = dBlog * dQ[(iIndice*3)+2]
        dRisultato = Math.exp(-((dRscaled + dGscaled + dBscaled) - 255.0) * dLog255 / 255.0)
        #-------------------------------------------------------------
        if dRisultato > 255.0 :
            dRisultato = 255.0
        #-------------------------------------------------------------
        if iIndice == 0 :
            bRed1 = dRisultato
        elif iIndice == 1 :
            bGreen1 = dRisultato
        else:
            bBlu1 = dRisultato
    return bRed1 , bGreen1 , bBlu1        
# ----------------------------------------------------------------------------- 
# End Function : ConvertPixel()
# -----------------------------------------------------------------------------
# 
# ----------------------------------------------------------------------------- 
# Function : InversionMatrix 
# -----------------------------------------------------------------------------
def InversionMatrix(dCosX,dCosY,dCosZ):
    #-------------------------------------------------------------
    dA = dCosY[1] - ((dCosX[1] * dCosY[0]) / dCosX[0])
    dV = dCosZ[1] - ((dCosX[1] * dCosZ[0]) / dCosX[0])
    dC = dCosZ[2] - (dCosY[2] * dV/dA) + (dCosX[2] * ((((dV/dA) * dCosY[0]) / dCosX[0]) - (dCosZ[0] / dCosX[0])))    
    #-------------------------------------------------------------
    #matrix inversion
    #-------------------------------------------------------------
    dQ = [0,0,0,0,0,0,0,0,0]
    dQ[2] = (-dCosX[2] / dCosX[0] - dCosX[2] / dA * dCosX[1] / dCosX[0] * dCosY[0] / dCosX[0] + dCosY[2] / dA * dCosX[1]/ dCosX[0]) / dC  
    dQ[1] = -dQ[2] * dV / dA - dCosX[1] / (dCosX[0] * dA)
    dQ[0] = 1.0 / dCosX[0] - dQ[1] * dCosY[0] / dCosX[0] - dQ[2] * dCosZ[0] / dCosX[0]
    dQ[5] = (-dCosY[2] / dA + dCosX[2] / dA * dCosY[0] / dCosX[0]) / dC
    dQ[4] = -dQ[5] * dV / dA + 1.0 / dA
    dQ[3] = -dQ[4] * dCosY[0] / dCosX[0] - dQ[5] * dCosZ[0] / dCosX[0]
    dQ[8] = 1.0 / dC
    dQ[7] = -dQ[8] * dV / dA
    dQ[6] = -dQ[7] * dCosY[0] / dCosX[0] - dQ[8] * dCosZ[0] / dCosX[0]  
    return dQ
# ----------------------------------------------------------------------------- 
# End Function : InversionMatrix()
# -----------------------------------------------------------------------------
# 
# ----------------------------------------------------------------------------- 
# Function : OutputChannelsNames 
# name of the new channel (skeleton storage)
# OUTPUT_CH_NAME_S1 = "Hema"  
# OUTPUT_CH_NAME_S2 = "Dab"  
# OUTPUT_CH_NAME_S3 = ""  
# STAINING 
# -----------------------------------------------------------------------------
def OutputChannelsNames(stain):
    # -------------------------------------------------------------------------
    StainName1 = "None1" 
    StainName2 = "None2" 
    StainName3 = "None3" 
    # -------------------------------------------------------------------------
    if stain == 0:       #HEMA_DAB:
        StainName1 = "Hema"
        StainName2 = "Dab"
        StainName3 = "Null"
    elif stain == 1:     #HEMA_EOSIN:
        StainName1 = "Hema"
        StainName2 = "Eosin"
        StainName3 = "Null"
    elif stain == 2 :    #HEMA_EOSIN2:
        StainName1 = "Hema"
        StainName2 = "Eosin"
        StainName3 = "Null"
    elif stain == 3 :    #FEULGEN_LGRENN:
        StainName1 = "Feulgen"
        StainName2 = "LGreen"
        StainName3 = "Null"        
    elif stain == 4 :    #HEMA_EOSIN_DAB:
        StainName1 = "Hema"
        StainName2 = "Eosin"
        StainName3 = "Dab"         
    # -------------------------------------------------------------------------
    return  StainName1,StainName2,StainName3   
# ----------------------------------------------------------------------------- 
# End Function : OutputChannelsNames()
# -----------------------------------------------------------------------------
#
# -------------------------------------------------------------------------- 
# Function : ComputeTile 
# the global variable TILE_SIZE will be set to the TILESIZE_X,TILESIZE_Y
# size if the document is bigger than these values
# -------------------------------------------------------------------------
def ComputeTile(imageset,Tilesize_X,Tilesize_Y):
  # -----------------------------------------------------------------------
  TilesizeX = Tilesize_X
  TilesizeY = Tilesize_Y
  if TilesizeX < 2048:
    TilesizeX = 2048
  if TilesizeY < 2048:
    TilesizeY = 2048
  TILE_SIZE = core.Size2D(TilesizeX,TilesizeY)  
  # -----------------------------------------------------------------------
  if imageset is None:
    print ("Error [imageset]: ComputeTile"  )
    return TILE_SIZE
  # -----------------------------------------------------------------------
  planeRect = imageset.get_bounding_rectangle()
  if abs(planeRect.width - planeRect.x) > TILESIZE_X : 
    TILE_SIZE.width = TILESIZE_X
  if abs(planeRect.height - planeRect.y) > TILESIZE_Y : 
    TILE_SIZE.height = TILESIZE_Y
  # -----------------------------------------------------------------------
  return TILE_SIZE      
# -------------------------------------------------------------------------
# End Function : ComputeTile 
# ------------------------------------------------------------------------- 
#  
# ----------------------------------------------------------------------------- 
# Function : ConvertChannels 
# -----------------------------------------------------------------------------
def ConvertChannels(imageset,currentTileRect,plane,frame,Input,Output,dQ,Coefficent):
  # -------------------------------------------------------------------------
  if imageset is None :
    print ("Error [imageset]: [ConvertChannels]" )
    return False  
  if len(Input)!=3:
    print ("Error [input channels number]: [ConvertChannels]" )
    return False      
  if len(Output)!=3:
    print ("Error [output channels number]: [ConvertChannels]" )
    return False   
  # -------------------------------------------------------------------------
  pixelType = imageset.get_pixeltype()
  # -------------------------------------------------------------------------
  pixelCount = currentTileRect.width * currentTileRect.height
  RectFull = imageset.get_bounding_rectangle()
  # ---------------------------------------------------------------------------
  # buffer allocation - one for each channel
  # ---------------------------------------------------------------------------
  inputBufferCh1 = core.Buffer(pixelType, pixelCount)		
  inputBufferCh2 = core.Buffer(pixelType, pixelCount)		
  inputBufferCh3 = core.Buffer(pixelType, pixelCount)		  
  # ---------------------------------------------------------------------------
  # RESULT BUFFERS	
  # ---------------------------------------------------------------------------
  outputBufferCh1 = core.Buffer(pixelType, pixelCount)	
  outputBufferCh2 = core.Buffer(pixelType, pixelCount)	
  outputBufferCh3 = core.Buffer(pixelType, pixelCount)		
  # ---------------------------------------------------------------------------
  # Get the channels data
  # ---------------------------------------------------------------------------
  imageset.get_channeldata(RectFull, currentTileRect, inputBufferCh1, Input[0], plane, frame)
  imageset.get_channeldata(RectFull, currentTileRect, inputBufferCh2, Input[1], plane, frame)
  imageset.get_channeldata(RectFull, currentTileRect, inputBufferCh3, Input[2], plane, frame)  
  # ---------------------------------------------------------------------------
  # iterate through buffer and compute each pixel
  # ---------------------------------------------------------------------------
  for index in range(0, pixelCount):
      bRed = inputBufferCh1[index]
      bGreen = inputBufferCh2[index]      
      bBlu = inputBufferCh3[index]         
      bRed , bGreen , bBlu = ConvertPixel(bRed , bGreen , bBlu, dQ)
      # -----------------------------------------------------------------------
      if INVERT_IMAGE == True:
          bRed = Coefficent - round(bRed)
          bGreen = Coefficent - round(bGreen)
          bBlu = Coefficent - round(bBlu)
      # -----------------------------------------------------------------------
      outputBufferCh1[index] = round(bRed)
      outputBufferCh2[index] = round(bGreen)
      outputBufferCh3[index] = round(bBlu )     
     # ------------------------------------------------------------------------
     # output the results
  # ---------------------------------------------------------------------------
  imageset.set_channeldata(currentTileRect, outputBufferCh1, Output[0], plane, frame) 
  imageset.set_channeldata(currentTileRect, outputBufferCh2, Output[1], plane, frame) 
  imageset.set_channeldata(currentTileRect, outputBufferCh3, Output[2], plane, frame)   
  # ---------------------------------------------------------------------------
  return True  
# ----------------------------------------------------------------------------- 
# End Function : ConvertChannels()
# -----------------------------------------------------------------------------
#
# -----------------------------------------------------------------------------
# Function : ConvertChannelslTile
# -----------------------------------------------------------------------------
def ConvertChannelslTile(Orimageset,Input,Output,TILE_SIZE,dQ,Coefficent):
  # ------------------------------------------------------------------------------
  if Orimageset is None:
    print ("Error [imageset]: MA_Copy_Channel_with_tile"  )
    return False
  # --------------------------------------------------------------------------- 
  #print ("MA_Copy_Channel_with_tile" )          # DEBUG)
  print ("source channel : " + str(Input))
  print ("destination channel : " + str(Output))
  # ---------------------------------------------------------------------------  
  planeRect = Orimageset.get_bounding_rectangle()      
  frames = Orimageset.get_timepoint_count()
  planes = Orimageset.get_plane_count(0)        # <---TIMEPOINT
  # -------------------------------------------------------------------------
  # determine number of tiles in x and y - this can be either rect.size/TILE_SIZE or rect.size/TILE_SIZE + 1
  # -------------------------------------------------------------------------
  countX1 = (planeRect.width / TILE_SIZE.width) if (planeRect.width % TILE_SIZE.width == 0) else (planeRect.width / TILE_SIZE.width + 1)
  countY1 = (planeRect.height / TILE_SIZE.height) if (planeRect.height % TILE_SIZE.height == 0) else (planeRect.height / TILE_SIZE.height + 1)
  countX = int(countX1)
  countY = int(countY1)
  print ("countX : " + str(countX) + " countY : " + str(countY) )
  # ------------------------------------------------------------------------- 
  for frame in range(0, frames) :
    # -------------------------------------------------------------------------  
    for plane in range(0, planes) :
      # ----------------------------------------------------------------------- 
      print ("plane : " + str(plane) + " frame : " + str(frame) )
      # -----------------------------------------------------------------------  
      for indexY in range(0, countY) :
        # ---------------------------------------------------------------------   
        for indexX in range(0, countX) :
          # -------------------------------------------------------------------  
          print ("indexY : " + str(indexY) + " indexX : " + str(indexX) )          
          # -------------------------------------------------------------------   
          currentTileRect = core.Bounds2D(indexX * TILE_SIZE.width, indexY * TILE_SIZE.height, TILE_SIZE.width, TILE_SIZE.height)
          currentTileRect.translate(planeRect.get_upper_left())
          currentTileRect.intersect(planeRect)              
          # -------------------------------------------------------------------    
          #  perform the operation on the buffer
          # ------------------------------------------------------------------- 
          ConvertChannels(Orimageset,currentTileRect,plane,frame,Input,Output,dQ,Coefficent)
          # -------------------------------------------------------------------
  return True  
# -------------------------------------------------------------------------- 
# End Function : ConvertChannelslTile
# --------------------------------------------------------------------------  
#
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@  Main  @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
#
# ------------------------------ Script body ----------------------------------
# MAIN BODY
# -----------------------------------------------------------------------------    
# helper to get execution time
startTime = time.time()
# -----------------------------------------------------------------------------  
print ("Script is running ........ " )
#
# ----------------------------------------------------------------------------- 
# Skeleton & End/Points section
# ----------------------------------------------------------------------------- 
viewer,imageset = GetEnviroment()          
# -----------------------------------------------------------------------------          
if imageset is not None :
  # ---------------------------------------------------------------------------
  #  Image must be 8bit
  #  change here the active Staining 
  # ---------------------------------------------------------------------------
  CurrentStaining = STAINING  
  # ---------------------------------------------------------------------------
  dt,typeP,Coefficent = GetPixelTypes(imageset)  
  if Coefficent!=255:
      print ("Image must have 8bit depth (255) : " + str(Coefficent))  
  else:
      channels = imageset.get_channel_count() 
      if channels<3:  
          print ("Image must have at least 3 channels (RGB) : " + str(channels))    
      else:
          # ---------------------------------------------------------------
          outCh = []
          StainName1,StainName2,StainName3 = OutputChannelsNames(CurrentStaining)
          outCh.append(PrepareChannel(imageset,StainName1) )
          outCh.append(PrepareChannel(imageset,StainName2) )
          outCh.append(PrepareChannel(imageset,StainName3))           
          # ---------------------------------------------------------------              
          inCh = [INPUT_CHANNEL_R,INPUT_CHANNEL_G,INPUT_CHANNEL_B]
          # ---------------------------------------------------------------
          Stain1,Stain2,Stain3 = StainMatrix(CurrentStaining)
          dCosX,dCosY,dCosZ,dLen = NormalizeVectorLength(Stain1,Stain2,Stain3)
          dCosX,dCosY,dCosZ = TranslationMatrix(dCosX,dCosY,dCosZ)
          dQ = InversionMatrix(dCosX,dCosY,dCosZ)
          # ---------------------------------------------------------------
          ConvertChannelslTile(imageset,inCh,outCh,TILE_SIZE,dQ,Coefficent)
          # ---------------------------------------------------------------
else:
    print("Imageset not open....")              
# ---------------------------------------------------------------------------- 
endTime = time.time()
print ("script time: " + str(endTime - startTime))
# ------------------------------ Script body ---------------------------------- 
# End of main body
# -----------------------------------------------------------------------------   


