# coding: utf-8
#------------------------------------------------------------------------------
#
# NAME: Matrioshka Doll 
# FILE: Matrioshka_Doll_Rev1_1_05
# REVISION : 1.06  - 2021-07-13
# AUTHOR : Maurizio Abbate
# Copyright(c) 2021 arivis AG, Germany. All Rights Reserved.
#
# Permission is granted to use, modify and distribute this code,
# as long as this copyright notice remains part of the code.
#
#
# PURPOSE : starting from a single/multiple objects, the script build 
# a set of concentric segments centered on the source object centroid 
# MATRYOSHKA_DOLL concept
#
# PUBLIC VARIABLES DESCRIPTION:
#           MATRYOSHKA_DOLL_NUM = sets the numbers of the concentric segments
#                                 [NUMBER - INTEGER]
#           TAG_DESCRIPTOR = set the TAG from which the original 
#                            segments have been stored [TEXT - STRING]
#           APPLY_CONVEX_HULL = the segment borders are corrected using 
#                               the convex hull algorithm [BOOL - BOOLEAN]
#           MATRYOSHKA_WITH_HOLES = Enable the holes creation inside each Doll 
#
# NOTES: Tested on 3.4
#
# ------------------------------ External Package Import ----------------------
import math as Math
import time
import arivis
import arivis_core as core
import arivis_objects as objects
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ USER SETTINGS @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@
TAG_DESCRIPTOR = "Dilate" #TAG labeling the main segment used to create the Dolls
#
TAG_SCRIPT_DESCRIPTOR = "Matrioshka_"
# total number of dolls including the original segment
MATRYOSHKA_DOLL_NUM = 2        # Max value == 10 
#
APPLY_CONVEX_HULL = True
#
MATRYOSHKA_WITH_HOLES = True 
# @@@@@@@@@@@@@@@@@@@@@@@@@@@@@@ END OF USER SETTINGS @@@@@@@@@@@@@@@@@@@@@@@@@
#
# ------------------------------ Global variables  ----------------------------
#   DON'T MODIFY the following variables
# -----------------------------------------------------------------------------
FEATUREDESCRIPTOR_NAME = "Source ID"
FEATUREVALUE_NAME = "Source Segment ID"
REF_ZOOM = 1.0
#
# ------------------------------ End Global variables  ------------------------
#
# ------------------------------ Script body ---------------------------------- 
# Function : Get_Enviroment
# -----------------------------------------------------------------------------
def MAGetEnviroment():
  # ---------------------------------------------------------------------------
  # return the viewer and the imageset  objects
  # ---------------------------------------------------------------------------
  viewer = arivis.App.get_active_viewer()
  imageset = viewer.get_imageset()
  if None == imageset :
    print( "No Image Set open" )
    return viewer,None  
  # ------------------------------------------------------------------------------ 
  return viewer,imageset
# ------------------------------ Script body ---------------------------------- 
# End Function : MAGetEnviroment
# ----------------------------------------------------------------------------
# ------------------------------ Script body ---------------------------------
# Function : MAGetStorage
# ----------------------------------------------------------------------------
def MAGetStorage(imageset):
  # --------------------------------------------------------------------------
  # return the storage object
  # --------------------------------------------------------------------------
  if None == imageset :
    print( "No Image Set open" )
    return None  
  # --------------------------------------------------------------------------
  document = imageset.get_document()
  if None == document :
    print( "No Document open" )
    return None
  # --------------------------------------------------------------------------
  store = document.get_store(imageset, objects.Store.DOCUMENT_STORE)
  if None == store :
    print( "No Measurements availble" )
    return None
  # --------------------------------------------------------------------------
  # Get the full object's ID list - if the list's lenght is == 0 no storage 
  # --------------------------------------------------------------------------
  IdList = store.get_object_ids("")
  #print( "lenght " + str(len(IdList))   )
  #---------------------------------------------------------------------------  
  if 0 == len(IdList):
    print( "No Objects" )
    return None
  # --------------------------------------------------------------------------
  #print( "Get_Storage - OK " )
  return store
# ------------------------------ Script body ---------------------------------
# End Function : MAGetStorage
# ----------------------------------------------------------------------------
# ------------------------------ Script body --------------------------------- 
# Function : GetObject
# ----------------------------------------------------------------------------
def GetObject(store,IdOBJ):
  # --------------------------------------------------------------------------
  # return the object centroid (3D_Bounding Box - pixels)
  # NO_Error , Point3D(X ,Y, Z)
  # strTAG
  # --------------------------------------------------------------------------
  if None == store :
   print( "No Measurements available" )
   return None 
  # --------------------------------------------------------------------------
  object1 = store.get_object(IdOBJ,True) 
  if None == object1:  
    # ------------------------------------------------------------------------
    print( "No Segment available" )
    return None
  else:
    return object1
# ------------------------------ Script body --------------------------------- 
# End Function : GetObject
# ---------------------------------------------------------------------------- 
# ------------------------------ Script body --------------------------------- 
# Function : GetObjectBBCentroid
# ----------------------------------------------------------------------------
def GetObjectBBCentroid(store,IdOBJ,strTAG):
  # --------------------------------------------------------------------------
  # return the object centroid (3D_Bounding Box - pixels)
  # NO_Error , Point3D(X ,Y, Z)
  # strTAG
  # --------------------------------------------------------------------------
  FEATURE_BB_CENTER = "Center of Bounding Box"
  SUB_FEATURE_BB_CENTER_NUM  = 6
  SUB_FEATURE_BB_XP  = 3
  SUB_FEATURE_BB_YP  = 4
  SUB_FEATURE_BB_ZP  = 5
  # --------------------------------------------------------------------------
  if None == store :
   print( "No Measurements available" )
   return core.Point3D(-1,-1,-1)       
  # --------------------------------------------------------------------------
  object1 = store.get_object(IdOBJ,False) 
  if None != object1:  
    # ------------------------------------------------------------------------
    TAgObject1 = object1.get_tags()
    #print( " TAGs " + str(TAgObject1) + " ID " + str(IDobject1))
    if strTAG in TAgObject1:
      # ---------------------------------------------------------------------- 
      #BB4d = object1.get_bounds()       # 4D bounding box  
      #print( "Bounds " + str(BB4d))
      # ----------------------------------------------------------------------
      # get_feature_descriptor - "Center of Bounding Box"
      #     FeatureName = FeatureDesc.get_name()
      #     print( "FeatureDesc " + FeatureName)
      #     featureID = FeatureDesc.get_id()
      #     print( "FeatureId " + str(featureID))
      # ----------------------------------------------------------------------  
      FeatureDesc = store.get_feature_descriptor(FEATURE_BB_CENTER)
      # ----------------------------------------------------------------------
      #     get_feature_descriptor - "Center of Bounding Box" 
      #     single features (XYZ, X(pix),Y(pix),Z(pix)
      #     objects.ValueDescriptor.TYPE_FLOAT
      # -----------------------------------------------------------------------   
      FeatureValue = FeatureDesc.get_values()
      #print( "FeatureValue " + str(FeatureValue))
      #print( "lenght " + str(len(FeatureValue))  )
      if SUB_FEATURE_BB_CENTER_NUM == len(FeatureValue):
        # ---------------------------------------------------------------------          
        FeatureValue1 = FeatureValue[SUB_FEATURE_BB_XP].get_name()
        FeatureValue2 = FeatureValue[SUB_FEATURE_BB_YP].get_name()
        FeatureValue3 = FeatureValue[SUB_FEATURE_BB_ZP].get_name()
        #print( "FeatureValue1 " + FeatureValue1 )
        #FeatureValueType = FeatureValue[SUB_FEATURE_BB_XP].get_type()
        #print( "FeatureValueType " + str(FeatureValueType))
        #
        Feature1 = store.get_feature_for_object(FeatureDesc,object1)
        #
        testo = str(Feature1.get_value(FeatureValue1)).replace(",", ".") 
        Featurevalue_X = int(float(testo))
        testo = str(Feature1.get_value(FeatureValue2)).replace(",", ".")         
        Featurevalue_Y = int(float(testo))
        testo = str(Feature1.get_value(FeatureValue3)).replace(",", ".")  
        Featurevalue_Z = int(float(testo))
        # --------------------------------------------------------------------
        return core.Point3D(Featurevalue_X,Featurevalue_Y,Featurevalue_Z)
        # --------------------------------------------------------------------
      else:
        print( "Error SUB_FEATURE_BB_CENTER_NUM " + str(len(FeatureValue)))
        return core.Point3D(-1,-1,-1)     
      # ----------------------------------------------------------------------
    else:
      print( "The tag " + strTAG + " does not exist, using " + str(TAgObject1[0]) + " instead")
      return core.Point3D(-1,-1,-1)
# ------------------------------ Script body --------------------------------- 
# End Function : GetObjectBBCentroid
# ----------------------------------------------------------------------------
# ------------------------------ Script body --------------------------------
# Function : GetObjectPlaneRange
# ----------------------------------------------------------------------------
def GetObjectPlaneRange(store,IdOBJ,strTAG):
  # --------------------------------------------------------------------------
  # return : the object first / last  planes 
  # order  : Point2D(first ,last)
  # ------------------------------------------------------------------------------
  FEATURE_PLANES = "Plane"
  SUB_FEATURE_PLANES_NUM  = 3
  SUB_FEATURE_PLANES_FIRST  = 0
  SUB_FEATURE_PLANES_LAST = 1
  #SUB_FEATURE_PLANES_COUNT  = 2
  # --------------------------------------------------------------------------
  if None == store :
   print( "No Measurements available" )
   return core.Point2D(-1,-1)    
  # --------------------------------------------------------------------------
  object1 = store.get_object(IdOBJ,True) 
  if None != object1:  
    # ------------------------------------------------------------------------
    TAgObject1 = object1.get_tags()
    if strTAG in TAgObject1:
      # ----------------------------------------------------------------------  
      FeatureDesc = store.get_feature_descriptor(FEATURE_PLANES)
      # ----------------------------------------------------------------------
      #     get_feature_descriptor - "Center of Bounding Box" 
      #     single features (XYZ, X(pix),Y(pix),Z(pix)
      #     objects.ValueDescriptor.TYPE_FLOAT
      # ------------------------------------------------------------------------------    
      FeatureValue = FeatureDesc.get_values() 
      if SUB_FEATURE_PLANES_NUM == len(FeatureValue):
        # ------------------------------------------------------------------------------   
        # ATTENZIONE: il conteggio piani parte da 0         
        # ------------------------------------------------------------------------------   
        FeatureValue1 = FeatureValue[SUB_FEATURE_PLANES_FIRST].get_name()
        FeatureValue2 = FeatureValue[SUB_FEATURE_PLANES_LAST].get_name()
        #FeatureValue3 = FeatureValue[SUB_FEATURE_PLANES_COUNT].get_name()
        Feature1 = store.get_feature_for_object(FeatureDesc,object1)
        testo = str(Feature1.get_value(FeatureValue1)).replace(",", ".") 
        Featurevalue_X = int(float(testo))
        testo = str(Feature1.get_value(FeatureValue2)).replace(",", ".")         
        Featurevalue_Y = int(float(testo))
        #testo = str(Feature1.get_value(FeatureValue3)).replace(",", ".")  
        #Featurevalue_Z = int(float(testo))
        # ------------------------------------------------------------------------------ 
        return core.Point2D(Featurevalue_X - 1,Featurevalue_Y)
        # ------------------------------------------------------------------------------ 
      else:
        print( "Error SUB_FEATURE_PLANES_NUM " + str(len(FeatureValue))     ) 
        return core.Point2D(-1,-1) 
      # ------------------------------------------------------------------------------
    else:
      print( "The tag " + strTAG + " does not exist, using " + str(TAgObject1[0]) + " instead" )
      return core.Point2D(-1,-1) 
# ------------------------------ Script body ----------------------------------
# End Function : GetObjectPlaneRange
# -----------------------------------------------------------------------------
# ------------------------------ Script body ----------------------------------
# Function : GetObjectBBSize
# -----------------------------------------------------------------------------
def GetObjectBBSize(store,IdOBJ,strTAG):
  # ------------------------------------------------------------------------------
  # return : the object BB dimensions (3D_Bounding Box - pixels)
  # order  : Point3D(sizeX ,sizeY, sizeZ)
  # ------------------------------------------------------------------------------
  FEATURE_BB_SIZE = "Bounding Box (Pixel)"
  SUB_FEATURE_BB_SIZE_NUM  = 9
  SUB_FEATURE_BB_SIZEXP  = 6
  SUB_FEATURE_BB_SIZEYP  = 7
  SUB_FEATURE_BB_SIZEZP  = 8
  # ------------------------------------------------------------------------------
  if None == store :
   print( "No Measurements available" )
   return core.Point3D(-1,-1,-1)     
  # ------------------------------------------------------------------------------
  object1 = store.GetObject(IdOBJ,True) 
  if None != object1:  
    # ------------------------------------------------------------------------------
    TAgObject1 = object1.get_tags()
    if strTAG in TAgObject1:
      # ------------------------------------------------------------------------------    
      FeatureDesc = store.get_feature_descriptor(FEATURE_BB_SIZE)
      # ------------------------------------------------------------------------------
      #     get_feature_descriptor - "Center of Bounding Box" - single features (XYZ, X(pix),Y(pix),Z(pix)
      #     objects.ValueDescriptor.TYPE_FLOAT
      # ------------------------------------------------------------------------------    
      FeatureValue = FeatureDesc.get_values()
      if SUB_FEATURE_BB_SIZE_NUM == len(FeatureValue):
        # ------------------------------------------------------------------------------            
        FeatureValue1 = FeatureValue[SUB_FEATURE_BB_SIZEXP].get_name()
        FeatureValue2 = FeatureValue[SUB_FEATURE_BB_SIZEYP].get_name()
        FeatureValue3 = FeatureValue[SUB_FEATURE_BB_SIZEZP].get_name()
        Feature1 = store.get_feature_for_object(FeatureDesc,object1)
        testo = str(Feature1.get_value(FeatureValue1)).replace(",", ".") 
        Featurevalue_X = int(float(testo))
        testo = str(Feature1.get_value(FeatureValue2)).replace(",", ".")         
        Featurevalue_Y = int(float(testo))
        testo = str(Feature1.get_value(FeatureValue3)).replace(",", ".")  
        Featurevalue_Z = int(float(testo))
        # ------------------------------------------------------------------------------ 
        return core.Point3D(Featurevalue_X,Featurevalue_Y,Featurevalue_Z)
        # ------------------------------------------------------------------------------ 
      else:
        print( "Error SUB_FEATURE_BB_SIZE_NUM " + str(len(FeatureValue))      )
        return core.Point3D(-1,-1,-1)
      # ------------------------------------------------------------------------------
    else:
      print( "The tag " + strTAG + " does not exist, using " + str(TAgObject1[0]) + " instead" )
      return core.Point3D(-1,-1,-1)
# ------------------------------ Script body ----------------------------------
# End Function : GetObjectBBCentroid
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body --------------------------------- 
# Function : CreateCustomFeature
# ----------------------------------------------------------------------------
def CreateCustomFeature(store,description,name):
  # --------------------------------------------------------------------------
  if store is None:
    print( "No Measurements available" )
    return False    
  # --------------------------------------------------------------------------
  # check if the feature descriptor exist
  # --------------------------------------------------------------------------  
  bCrea = False
  ListDescriptors = store.get_feature_descriptors()  
  for A in ListDescriptors:
      if A.get_name() == description:
          featureDescriptor = A
          break
      else:    
          # -------------------------------------------------------------------
          # create feature descriptor with multiple values of different types
          # -------------------------------------------------------------------
          featureDescriptor = objects.FeatureDescriptor()
          featureDescriptor.set_name(description)        #FEATUREDESCRIPTOR_NAME
          bCrea = True
  # --------------------------------------------------------------------------   CreateCustomFeature(store,FEATUREDESCRIPTOR_NAME,FEATUREVALUE_NAME)
  valDesc = objects.ValueDescriptor()
  valDesc.set_name(name)       #FEATUREVALUE_NAME
  valDesc.set_type(objects.ValueDescriptor.TYPE_FLOAT)
  featureDescriptor.add_value(valDesc)
  # --------------------------------------------------------------------------
  # add feature to object store
  # --------------------------------------------------------------------------
  if bCrea == True:
      if False == store.create_stored_feature(featureDescriptor):
          print( "feature NOT added (Already existing) " + featureDescriptor.get_name())
      else:    
          print( "feature added: " + featureDescriptor.get_name())
  return True
# ------------------------------ Script body --------------------------------- 
# End Function : CreateCustomFeature
# ----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : CreateMySectorFeature
# -----------------------------------------------------------------------------
def CreateMySectorFeature(store):
  #get object store
  #objectStore = GetObject_store()
  # ---------------------------------------------------------------------------
  if None == store :
    print( "No Measurements available" )
    return False    
  # ---------------------------------------------------------------------------
  # create feature descriptor with multiple values of different types
  # FEATUREDESCRIPTOR_NAME = "Source ID"
  # FEATUREVALUE_NAME = "Source Segment ID"
  # ---------------------------------------------------------------------------
  featureDescriptor = objects.FeatureDescriptor()
  featureDescriptor.set_name(FEATUREDESCRIPTOR_NAME)
  # ---------------------------------------------------------------------------
  valDesc = objects.ValueDescriptor()
  valDesc.set_name(FEATUREVALUE_NAME)
  valDesc.set_type(objects.ValueDescriptor.TYPE_FLOAT)
  featureDescriptor.add_value(valDesc)
  # ---------------------------------------------------------------------------
  # add feature to object store
  store.create_stored_feature(featureDescriptor)
  print( "feature added: " + featureDescriptor.get_name())
  return True
# ------------------------------ Script body ----------------------------------
# End Function : CreateMySectorFeature
# ----------------------------------------------------------------------------
# ------------------------------ Script body ----------------------------------
# Function : InsertSourceID
# under test - delete
# -----------------------------------------------------------------------------
def InsertSourceID(store,SourceID,object):
  # ------------------------------------------------------------------------------
  if None == store :
    print( "No Measurements available" )
    return False    
  if None == object:
    print( "No Object")
    return False 
  # ------------------------------------------------------------------------------
  featureDesc = store.get_feature_descriptor(FEATUREDESCRIPTOR_NAME)
  feature = objects.Feature()
  feature.set_value(FEATUREVALUE_NAME,float(SourceID))
  store.set_feature_for_object(feature, featureDesc, object)  
  return True    
# ------------------------------ Script body ----------------------------------
# End Function : InsertSourceID
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : InsertSegmentInGroup
# -----------------------------------------------------------------------------
def InsertSegmentInGroup(store,NewGroup,object1):
  # ------------------------------------------------------------------------------
  #if None == NewGroup :
  #  print( "No Group available" )
  #  return False    
  # ------------------------------------------------------------------------------
  if None == store :
    print( "No Measurements available" )
    return False      
  # ------------------------------------------------------------------------------
  if None != object1:  
    # ------------------------------------------------------------------------------
    NewGroup.add_object(object1)
  return True    
# ------------------------------ Script body ----------------------------------
# End Function : InsertSegmentInGroup
# ----------------------------------------------------------------------------
#
# ---------------------------------------------------------------------------- 
# Function : RemoveTag    
# ----------------------------------------------------------------------------
def RemoveTag(objectStore,Tag):
  # --------------------------------------------------------------------------
  list = objectStore.get_object_ids(Tag) 
  for X in range(0,len(list)):
    myObject = objectStore.get_object(list[X],True) 
    myObject.remove_tag(Tag) #lets name the list so we can find it easily!
    objectStore.update_object(myObject) #make it so   
# -------------------------------------------------------------------------
# End Function : RemoveTag
# -------------------------------------------------------------------------   
#
# ------------------------------ Script body ----------------------------------
# Function : MAEstimateCoef
# -----------------------------------------------------------------------------
def MAEstimateCoef(Outer, Inner): 
    # ------------------------------------------------------------------------------
    # x = Y = core.Point2D
    # number of observations/points 
    # ------------------------------------------------------------------------------    
    SizeX = Outer.y - Outer.x 
    SizeY = Inner.y - Inner.x    
    #print( "SizeX :" + str(SizeX) + "  Outer.x :" + str(Outer.x) + "  Outer.y :" + str(Outer.y))
    if SizeX == 0:
      SizeX = 1
    if SizeY == 0:
      SizeY = 1
    # ------------------------------------------------------------------------------
    # Comute the mean of both vectors
    # ------------------------------------------------------------------------------      
    SumX = 0
    SumY = 0    
    # ------------------------------------------------------------------------------    
    for index in range(int(Outer.x),int(Outer.y)):
      SumX += index
    for index in range(int(Inner.x),int(Inner.y)):
      SumY += index      
    # ------------------------------------------------------------------------------    
    m_x = float(SumX)/float(SizeX)
    m_y = float(SumY)/float(SizeY)
    # ------------------------------------------------------------------------------    
    # calculating cross-deviation and deviation about x 
    # ------------------------------------------------------------------------------    
    SS_xy = (float(SumX) * float(SumY)) - (m_x * m_y)
    SS_xx = (float(SumX) * float(SumX)) - (float(SizeX)*m_x*m_x)
    if SS_xx == 0:
      SS_xx = 1
    # calculating regression coefficients 
    # ------------------------------------------------------------------------------    
    b_1 = SS_xy / SS_xx 
    b_0 = m_y - b_1*m_x 
    # ------------------------------------------------------------------------------      
    return(b_0, b_1) 
# ------------------------------ Script body ----------------------------------
# End Function : MAEstimateCoef
# ----------------------------------------------------------------------------- 
# ------------------------------ Script body ----------------------------------
# Function : MACross
# -----------------------------------------------------------------------------
def MACross(o, a, b):
  # ------------------------------------------------------------------------------
  #MACross(o, a, b):
  # 2D cross product of OA and OB vectors, i.e. z-component of their 3D cross product.
  # Returns a positive value, if OAB makes a counter-clockwise turn,
  # negative for clockwise turn, and zero if the points are collinear. 
  #return (a[0] - o[0]) * (b[1] - o[1]) - (a[1] - o[1]) * (b[0] - o[0])   
  # ------------------------------------------------------------------------------
  return (int(a.x)-int(o.x)) * (int(b.y)-int(o.y)) - (int(a.y)-int(o.y)) * (int(b.x)-int(o.x))
# ------------------------------ Script body ----------------------------------
# End Function : MACross
# ----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : MAConvexHull
# ----------------------------------------------------------------------------
def MAConvexHull(points):
  # ------------------------------------------------------------------------------
  #Computes the convex hull of a set of 2D points.
  #Input: an iterable sequence of (x, y) pairs representing the points.
  #Output: a list of vertices of the convex hull in counter-clockwise order,
  #starting from the vertex with the lexicographically smallest coordinates.
  #Implements Andrew's monotone chain algorithm. O(n log n) complexity.
  # Sort the points lexicographically (tuples are compared lexicographically).
  # Remove duplicates to detect the case we have just one unique point.
  # return : 
  # order  : 
  # modified by Fahimeh 2021-04-24
  # ------------------------------------------------------------------------------
  if False==APPLY_CONVEX_HULL or len(points) <= 1:
    return points  
  # ------------------------------------------------------------------------------
  # Build lower hull 
  # ------------------------------------------------------------------------------
  result = []
  start = points[0]
  min_x = start.x
  for p in points[1:]:
    if p.x < min_x:
      min_x = p.x
      start = p
  point = start
  result.append(start)
  far_point = None
  while far_point is not start:
    # get the first point (initial max) to use to compare with others
    p1 = None
    for p in points:
      if p is point:
          continue
      else:
          p1 = p
          break
    far_point = p1
    for p2 in points:
      # ensure we aren't comparing to self or pivot point
      if p2 is point or p2 is p1:
        continue
      else:
        direction = (((p2.x - point.x) * (far_point.y - point.y)) - ((far_point.x - point.x) * (p2.y - point.y)))
        if direction > 0:
            far_point = p2
    result.append(far_point)
    point = far_point
  return result
  # ---------------------------------------------------------------------------
  #print( "Lenght result " + str(len(result)))
# ------------------------------ Script body ----------------------------------
# End Function : MAConvexHull
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : MAGetOuterPolyline
# -----------------------------------------------------------------------------
def MAGetOuterPolyline(Oggetto,PlaneIndex):
  # ------------------------------------------------------------------------------
  # compute the scaled objects border
  # return : [polyline], # of remaining poligons  
  # Order  : [polyline] , #Poly
  # ----> delete
  # ------------------------------------------------------------------------------
  if None == Oggetto :
    print( "No Segment available" )
    return  None, 0    
  # ------------------------------------------------------------------------------
  Poligono = Oggetto.get_polygons(PlaneIndex)
  return Poligono,len(Poligono)
# ------------------------------ Script body ---------------------------------- 
# End Function : MAGetOuterPolyline
# ----------------------------------------------------------------------------- 
# ------------------------------ Script body ---------------------------------- 
# Function : MAScaleOuterPolyline
# -----------------------------------------------------------------------------
def MAScaleOuterPolyline(Bordo,Centroide,ZoomOuter):
  # ------------------------------------------------------------------------------
  # compute the scaled objects border
  # the local centroid is computed
  # return : polyline  
  # Order  : polyline
  # ------------------------------------------------------------------------------
  if 0>=Centroide.x or 0>=Centroide.y or 0>=Centroide.z :
    print( "Error: Centroide == 0 (MAScaleOuterPolyline)" )
    return  None  
  if None == Bordo :
    print( "No polyline available" )
    return  None    
  if 0 == len(Bordo) :
    print( "No polyline available" )
    return  None    
  # ------------------------------------------------------------------------------
  # local centroid
  # ------------------------------------------------------------------------------  
  MinX=Bordo[0].x
  MinY=Bordo[0].y
  MaxX=0
  MaxY=0   
  for index in range(0,len(Bordo)) :
    if MinX>Bordo[index].x:
      MinX = Bordo[index].x
    if MinY>Bordo[index].y:
      MinY = Bordo[index].y
    if MaxX<Bordo[index].x:
      MaxX = Bordo[index].x
    if MaxY<Bordo[index].y:
      MaxY = Bordo[index].y     
  centroX =  ((MaxX-MinX)/2) + MinX
  centroY =  ((MaxY-MinY)/2) + MinY  
  # ------------------------------------------------------------------------------        
  #print( "Centroide (ori.x ori.y local.x local.y) " + str(Centroide.x) + " " + str(Centroide.y) + " " + str(centroX)+ " " + str(centroY) )
  # ------------------------------------------------------------------------------          
 # BordoScaled  = [core.Point2D()] #core.Point2D
  #del BordoScaled[0]  
  BordoScaled  = []  
  for index in range(0,len(Bordo)) :
    x =(Bordo[index].x - centroX)*ZoomOuter + centroX
    y =(Bordo[index].y - centroY)*ZoomOuter + centroY    
    Punto = core.Point2D(int(x),int(y))
    BordoScaled.append(Punto)
    # ------------------------------------------------------------------------
  BordoScaled.append(BordoScaled[0])    
  #---------------------------------------------------------------------------   
  return BordoScaled
# ------------------------------ Script body ---------------------------------- 
# End Function : MAScaleOuterPolyline
# -----------------------------------------------------------------------------
#
# ------------------------------ Class body ----------------------------------- 
# Class : Holes
# -----------------------------------------------------------------------------       
class Holes:
    def __init__(self):
        self.Top = 0
        self.Bottom = 0
        self.BHoles = False
        self.Holeslist = [] # List[List[List[core.point2d]]]        
     # ----------------------------------------------------------------------- 
    def SetTopBottomPlane(self,t,b):
        self.Top = t
        self.Bottom = b  
        Zrange = abs(self.Bottom-self.Top) + 1
        tempList = []
        for i in range(0,Zrange):
            self.Holeslist.append(tempList)
     # ----------------------------------------------------------------------- 
    def SetTopPlane(self,t):
        self.Top = t
     # ----------------------------------------------------------------------- 
    def SetBottomPlane(self,b):
        self.Bottom = b  
     # ----------------------------------------------------------------------- 
    def GetBottomPlane(self):
        return(self.Bottom)  
     # ----------------------------------------------------------------------- 
    def GetTopPlane(self):
        return(self.Top)          
     # ----------------------------------------------------------------------- 
    def SetHoles(self,h):
        self.BHoles = h
    # ----------------------------------------------------------------------- 
    def GetHoles(self):
        return(self.BHoles)        
    # ----------------------------------------------------------------------- 
    def GetValidHoles(self,p):
        p1 = p - self.Top 
        if p1 >=0 and p1<= self.Bottom:
            if len(self.Holeslist[p1])==0:
                return(False)  
        return(True)  
    # -----------------------------------------------------------------------   
    def AddHoleslist__(self,x,p):
        p1 = p - self.Top
        if p1 >=0 and p1<= self.Bottom:        
            index = -1
            maxi = 0
            for i in range(0,len(x)):
                if len(x[i]) > maxi:
                    maxi = len(x[i])
                    index = i
            self.Holeslist[p1] = x[index]
    # -----------------------------------------------------------------------   
    def AddHoleslist(self,x):
        self.Holeslist.append(x) 
    # -------------------------------------------------------------------------   
    def RetriveHoleslist(self):
        return self.Holeslist
    # -------------------------------------------------------------------------  
    def RetrivePlanelist(self,p):
        if p >=0 and p<=abs(self.Top-self.Bottom):
            return self.Holeslist[p]     
   # -------------------------------------------------------------------------  
    def RetrivePlanelist___(self,p):
        p1 = p - self.Top
        if p1 >=0 and p1<= self.Bottom:
            return self.Holeslist[p1]      
        return None
# ------------------------------ End Class body ----------------------------------- 
# end Class : Holes
# -----------------------------------------------------------------------------    
#
# ------------------------------ Script body ---------------------------------- 
# Function : MAComputeSingleDoll
# -----------------------------------------------------------------------------
def MAComputeSingleDoll(Oggetto,Centroide,ZoomOuter,PlanesRange,ListOfPreviousHoles):
  # ---------------------------------------------------------------------------
  # compute the single objects from source object outline
  # return :   
  # Order  : segment
  # ATTENZIONE : index starts from 0
  # ---------------------------------------------------------------------------
  if None == Oggetto :
    print( "No segment available" )
    return  None,None  
  # ---------------------------------------------------------------------------   
  # get the time point associated to the source Object
  # ---------------------------------------------------------------------------   
  tp = Oggetto.get_timepoint()
  # ---------------------------------------------------------------------------    
  StepsOuter = int(float(PlanesRange.y - PlanesRange.x) * ZoomOuter) + 1
  # --------------------------------------------------------------------------
  PlanesRangeOuter = core.Point2D(Centroide.z - (int(StepsOuter/2)),Centroide.z + (int(StepsOuter/2)))
  # --------------------------------------------------------------------------- 
  # linear regression on Z planes
  # --------------------------------------------------------------------------- 
  B0_outer,B1_outer = MAEstimateCoef(PlanesRange,PlanesRangeOuter)
  # ---------------------------------------------------------------------------
  # ciclo sui piani
  # ---------------------------------------------------------------------------
  boxObject = objects.Segment()
  # ---------------------------------------------------------------------------
  ListOfHoles = Holes()
  ListOfHoles.SetTopBottomPlane(PlanesRange.x , PlanesRange.y)
  ListOfHoles.SetHoles(True)
  # ---------------------------------------------------------------------------  
  for PlaneIndex in range(PlanesRange.x,PlanesRange.y) :  
    # -------------------------------------------------------------------------   
    # Compute the outer polygon
    # -------------------------------------------------------------------------   
    deci, PlaneZ = Math.modf(B0_outer + B1_outer*PlaneIndex)
    if deci>=0.5:
      PlaneZ += 1 
    # ---------------------------------------------------------------------------  
    if PlaneZ<PlanesRange.x:
      PlaneZ=PlanesRange.x 
    if PlaneZ>PlanesRange.y:
      PlaneZ=PlanesRange.y   
    # -------------------------------------------------------------------------        
    BordoList,PCount = MAGetOuterPolyline(Oggetto,PlaneIndex)
    # -------------------------------------------------------------------------       
    ListPoint2D = []
    maxi = 0
    indexP = 0
    # ----------------------------------------------------------------------- 
    for indexPoly in range(0,PCount):
      # -----------------------------------------------------------------------   
      Bordo1 = BordoList[indexPoly].get_contour()
      if len(Bordo1) > maxi:
          maxi = len(Bordo1)
          indexP = indexPoly
    # ----------------------------------------------------------------------- 
    Bordo1 = BordoList[indexP].get_contour() 
    Bordo = MAConvexHull(Bordo1)      # <---- Convex Hull
    # ----------------------------------------------------------------------- 
    # BordoEsterno == List of core.Point2D
    # -----------------------------------------------------------------------       
    BordoEsterno = MAScaleOuterPolyline(Bordo,Centroide,ZoomOuter)
    ListPoint2D.append(BordoEsterno)
    # ----------------------------------------------------------------------- 
    if None!=BordoEsterno:
        # ---------------------------------------------------------------------             
        polygon = objects.Polygon()
        polygon.set_contour(BordoEsterno)     
        # --------------------------------------------------------------------                     
        if MATRYOSHKA_WITH_HOLES == True and True == ListOfPreviousHoles.GetHoles() and True == ListOfPreviousHoles.GetValidHoles(int(PlaneZ)):
            # -----------------------------------------------------------------                      
            holes = []
            hole = ListOfPreviousHoles.RetrivePlanelist___(int(PlaneZ))
            holes.append(hole)            
            if holes != None or len(holes)>0:                   
                polygon.set_holes(holes) 
        boxObject.add_polygon(polygon, int(PlaneZ)) 
        boxObject.set_timepoint(tp)
    # -------------------------------------------------------------------------   
    ListOfHoles.AddHoleslist__(ListPoint2D,int(PlaneZ))
    # -------------------------------------------------------------------------       
  return boxObject,ListOfHoles
# ------------------------------ Script body ----------------------------------
# End Function : MAComputeSingleDoll
# ----------------------------------------------------------------------------- 
#
# ------------------------------ Script body ---------------------------------- 
# Function : MAComputeLastDoll
# -----------------------------------------------------------------------------
def MAComputeLastDoll(Oggetto,PlanesRange,ListOfPreviousHoles):
  # ---------------------------------------------------------------------------
  # copy the last object  outline
  # return :   
  # Order  : segment
  # ATTENZIONE : index starts from 0
  # --------------------------------------------------------------------------
  if None == Oggetto :
    print( "No segment available [MAComputeLastDoll]" )
    return  None  
  # ---------------------------------------------------------------------------     
  # get the time point associated to the source Object
  # ---------------------------------------------------------------------------  
  tp = Oggetto.get_timepoint()
  # --------------------------------------------------------------------------   
  # ciclo sui piani
  # --------------------------------------------------------------------------   
  boxObject = objects.Segment()
  # ---------------------------------------------------------------------------  
  for PlaneIndex in range(PlanesRange.x,PlanesRange.y) :  
    # -------------------------------------------------------------------------   
    # Compute the outer polygon
    # -------------------------------------------------------------------------        
    BordoList,PCount = MAGetOuterPolyline(Oggetto,PlaneIndex)
    maxi = 0
    indexP = 0
    # ----------------------------------------------------------------------- 
    for indexPoly in range(0,PCount):
      # -----------------------------------------------------------------------   
      Bordo1 = BordoList[indexPoly].get_contour()
      if len(Bordo1) > maxi:
          maxi = len(Bordo1)
          indexP = indexPoly
    # ----------------------------------------------------------------------- 
    Bordo1 = BordoList[indexP].get_contour() 
    BordoEsterno = MAConvexHull(Bordo1)      # <---- Convex Hull
    # -----------------------------------------------------------------------   
    if None!=BordoEsterno:
        # ---------------------------------------------------------------------          
        polygon = objects.Polygon()
        polygon.set_contour(BordoEsterno) 
         # --------------------------------------------------------------------                     
        if MATRYOSHKA_WITH_HOLES == True and True == ListOfPreviousHoles.GetHoles() and True == ListOfPreviousHoles.GetValidHoles(PlaneIndex):
            # -----------------------------------------------------------------                      
            holes = []
            hole = ListOfPreviousHoles.RetrivePlanelist___(PlaneIndex)
            holes.append(hole)       
            if holes != None or len(holes)>0:
                polygon.set_holes(holes)   
        boxObject.add_polygon(polygon, PlaneIndex)  
        boxObject.set_timepoint(tp)
    # -------------------------------------------------------------------------
  return boxObject
# ------------------------------ Script body ----------------------------------
# End Function : MAComputeLastDoll
# -----------------------------------------------------------------------------
#
# ------------------------------ Script body ----------------------------------
# Function : MAComputeMatDoll
# -----------------------------------------------------------------------------
def MAComputeMatDoll(store,Oggetto,Centroide,PlanesRange,IdList):
  # --------------------------------------------------------------------------
  # compute the concentric objects 
  # return : Error  
  # Order  : NO_Error 
  # PlanesRange = # Object Z planes 
  # --------------------------------------------------------------------------
  if None == store :
    print( "No Measurements available" )
    return  False
  # --------------------------------------------------------------------------
  if None == Oggetto :
    print( "No Segment available" )
    return  False
  # --------------------------------------------------------------------------
  if MATRYOSHKA_DOLL_NUM < 1 :
    print( "Error: invalid MATRYOSHKA_DOLL_NUM (MAComputeMatDoll) : " + str(MATRYOSHKA_DOLL_NUM))
    return False
  # --------------------------------------------------------------------------
  # Loop to draw concentric objects
  # --------------------------------------------------------------------------  
  zoom = float(REF_ZOOM)/float(MATRYOSHKA_DOLL_NUM)
  
  
  #print( "Zoom -> " + str(zoom))
  # --------------------------------------------------------------------------   
  # creates dolls from inner to outer
  # --------------------------------------------------------------------------    
  ListOfHoles = Holes()       # <--- Class Holes
  for index in range(0,MATRYOSHKA_DOLL_NUM-1) :
    # ------------------------------------------------------------------------- 
    ZoomOuter = zoom * (index + 1)
    # ------------------------------------------------------------------------       
    boxObject,ListOfHoles__ = MAComputeSingleDoll(Oggetto,Centroide,ZoomOuter,PlanesRange,ListOfHoles)
    ListOfHoles = ListOfHoles__
    # ------------------------------------------------------------------------   
    # creo oggetto
    # TAG_SCRIPT_DESCRIPTOR
    # ------------------------------------------------------------------------ 
    if None !=  boxObject:
      # ---------------------------------------------------------------------- 
      SourceID = Oggetto.get_id()        
      boxObject.add_tag(TAG_SCRIPT_DESCRIPTOR + "_" + str(IdList)) 
      store.add_object(boxObject)   
      InsertSourceID(Store,SourceID,boxObject)
    # ------------------------------------------------------------------------
    print( "MATRYOSHKA -> " + str(index + 1) + " Obj ID -> " + str(IdList))
  # --------------------------------------------------------------------------   
  # Last doll has the same size of the main segment
  # --------------------------------------------------------------------------     
  boxObject = MAComputeLastDoll(Oggetto,PlanesRange,ListOfHoles)
  # --------------------------------------------------------------------------
  if None !=  boxObject:
     # -----------------------------------------------------------------------
     SourceID = Oggetto.get_id()        
     boxObject.add_tag(TAG_SCRIPT_DESCRIPTOR + "_" + str(IdList))  
     store.add_object(boxObject)   
     InsertSourceID(Store,SourceID,boxObject)
  # --------------------------------------------------------------------------
  print( "MATRYOSHKA -> " + str(MATRYOSHKA_DOLL_NUM) + " Obj ID -> " + str(IdList))
  # --------------------------------------------------------------------------
  RemoveTag(Store,"Script")  
  # --------------------------------------------------------------------------
  return True
# ------------------------------ Script body ---------------------------------
# End Function : MAComputeMatDoll
# ----------------------------------------------------------------------------
#
# ------------------------------ Script body ---------------------------------
# Main body
# ---------------------------------------------------------------------------- 
# helper to get execution time
startTime = time.time()
print ("MATRYOSHKA DOLLS creation starts......")
# ----------------------------------------------------------------------------- 
viewer,imageset = MAGetEnviroment()          
Store = MAGetStorage(imageset)
# -----------------------------------------------------------------------------           
if None != Store :
  # --------------------------------------------------------------------------- 
  CreateCustomFeature(Store,FEATUREDESCRIPTOR_NAME,FEATUREVALUE_NAME)
  #CreateMySectorFeature(Store)
  # --------------------------------------------------------------------------- 
  IdList = Store.get_object_ids(TAG_DESCRIPTOR)
  #---------------------------------------------------------------------------- 
  #  loop on ID list
  #----------------------------------------------------------------------------  
  #if 0 < int(len(IdList)):
  for indice in range(0,int(len(IdList))): 
    # -------------------------------------------------------------------------  
    Oggetto = GetObject(Store,IdList[indice])    
    print( "ID: " + str(IdList[indice]) + "  Obj index : " + str(indice) + "  # Objects: " + str(len(IdList)) + "  # TP: " + str(Oggetto.get_timepoint()))
    # -------------------------------------------------------------------------
    Centroide = GetObjectBBCentroid(Store,IdList[indice],TAG_DESCRIPTOR)
    # -------------------------------------------------------------------------
    PlanesRange = GetObjectPlaneRange(Store,IdList[indice],TAG_DESCRIPTOR)
    # -------------------------------------------------------------------------
    MAComputeMatDoll(Store,Oggetto,Centroide,PlanesRange,IdList[indice])
# -----------------------------------------------------------------------------          
endTime = time.time()
print( "time: " + str(endTime - startTime))
print ("MATRYOSHKA DOLLS creation ends......")
# ------------------------------ Script body ----------------------------------
# End of main body
# -----------------------------------------------------------------------------